// MIT License
//
// Copyright (c) 2017-2018 MessageKit
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
// SOFTWARE.

import Foundation
import UIKit

open class ContactMessageSizeCalculator: MessageSizeCalculator {
  // MARK: Open

  open override func messageContainerMaxWidth(for message: MessageType, at indexPath: IndexPath) -> CGFloat {
    let maxWidth = super.messageContainerMaxWidth(for: message, at: indexPath)
    let textInsets = contactLabelInsets(for: message)
    return maxWidth - textInsets.horizontal
  }

  open override func messageContainerSize(for message: MessageType, at indexPath: IndexPath) -> CGSize {
    let maxWidth = messageContainerMaxWidth(for: message, at: indexPath)

    var messageContainerSize: CGSize
    let attributedText: NSAttributedString

    switch message.kind {
    case .contact(let item):
      attributedText = NSAttributedString(string: item.displayName, attributes: [.font: contactLabelFont])
    default:
      fatalError("messageContainerSize received unhandled MessageDataType: \(message.kind)")
    }

    messageContainerSize = labelSize(for: attributedText, considering: maxWidth)

    let messageInsets = contactLabelInsets(for: message)
    messageContainerSize.width += messageInsets.horizontal
    messageContainerSize.height += messageInsets.vertical

    return messageContainerSize
  }

  open override func configure(attributes: UICollectionViewLayoutAttributes) {
    super.configure(attributes: attributes)
    guard let attributes = attributes as? MessagesCollectionViewLayoutAttributes else { return }
    attributes.messageLabelFont = contactLabelFont
  }

  // MARK: Public

  public var incomingMessageNameLabelInsets = UIEdgeInsets(top: 7, left: 46, bottom: 7, right: 30)
  public var outgoingMessageNameLabelInsets = UIEdgeInsets(top: 7, left: 41, bottom: 7, right: 35)
  public var contactLabelFont = UIFont.preferredFont(forTextStyle: .body)

  // MARK: Internal

  internal func contactLabelInsets(for message: MessageType) -> UIEdgeInsets {
    let dataSource = messagesLayout.messagesDataSource
    let isFromCurrentSender = dataSource.isFromCurrentSender(message: message)
    return isFromCurrentSender ? outgoingMessageNameLabelInsets : incomingMessageNameLabelInsets
  }
}
