// MIT License
//
// Copyright (c) 2017-2019 MessageKit
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in all
// copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
// SOFTWARE.

import UIKit

open class MessageContainerView: UIImageView {
  // MARK: Open

  open var style: MessageStyle = .none {
    didSet {
      applyMessageStyle()
    }
  }

  open override var frame: CGRect {
    didSet {
      sizeMaskToView()
    }
  }

  // MARK: Private

  // MARK: - Properties

  private let imageMask = UIImageView()

  // MARK: - Methods

  private func sizeMaskToView() {
    switch style {
    case .none, .custom:
      break
    case .bubble, .bubbleTail, .bubbleOutline, .bubbleTailOutline:
      imageMask.frame = bounds
    }
  }

  private func applyMessageStyle() {
    switch style {
    case .bubble, .bubbleTail:
      imageMask.image = style.image
      sizeMaskToView()
      mask = imageMask
      image = nil
    case .bubbleOutline(let color):
      let bubbleStyle: MessageStyle = .bubble
      imageMask.image = bubbleStyle.image
      sizeMaskToView()
      mask = imageMask
      image = style.image?.withRenderingMode(.alwaysTemplate)
      tintColor = color
    case .bubbleTailOutline(let color, let tail, let corner):
      let bubbleStyle: MessageStyle = .bubbleTail(tail, corner)
      imageMask.image = bubbleStyle.image
      sizeMaskToView()
      mask = imageMask
      image = style.image?.withRenderingMode(.alwaysTemplate)
      tintColor = color
    case .none:
      mask = nil
      image = nil
      tintColor = nil
    case .custom(let configurationClosure):
      mask = nil
      image = nil
      tintColor = nil
      configurationClosure(self)
    }
  }
}
